"use client";
import { AnnonceOtherImgProp } from "@/lib/types/AnnonceOhterImgProp";
import { useEffect, useState } from "react";
import Dropzone from "react-dropzone";
import { useRouter } from "next/navigation";

export default function EditAnnonceOtherImg({
    annonceId,
    annonceGallery,
}: {
    annonceId: number;
    annonceGallery: AnnonceOtherImgProp[] | undefined;
}) {
    const [originalImages, setOriginalImages] = useState<string[] | string[]>([]);
    const [pendingFiles, setPendingFiles] = useState<File[]>([]);
    const [uploading, setUploading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const router = useRouter();

    useEffect(() => {
        if (annonceGallery?.length) {
            const urls = annonceGallery.map((img) => `data:image/jpeg;base64,${img.imageData}`);
            setOriginalImages(urls);
        }
    }, [annonceGallery]);

    function onDrop(files: File[]) {
        if (!files?.length) return;
        const urls = files.map((f) => URL.createObjectURL(f));
        setPendingFiles((prev) => [...prev, ...files]);
    }

    async function handleUploadImages() {
        if (!pendingFiles.length) return;
        setUploading(true);
        setError(null);
        const fd = new FormData();
        pendingFiles.forEach((f) => fd.append("files", f));

        try {
            const res = await fetch(`/api/annonces/${annonceId}/other-images`, {
                method: "POST",
                body: fd,
            });
            if (!res.ok) {
                const err = await res.json().catch(() => ({}));
                throw new Error(err.error || "Échec de l'upload des images.");
            }
            setPendingFiles([]);
            router.push(`/annonces/${annonceId}`);
        } catch (e: any) {
            console.error(e);
            setError(e.message ?? "Erreur lors de l'upload des images.");
        } finally {
            setUploading(false);
        }
    }

    return (
        <div className="p-6">
            <h2 className="mb-4 text-2xl font-semibold">Modifier les autres images</h2>
            {originalImages.length > 0 ? (
                <div className="mb-6 grid grid-cols-2 gap-4 sm:grid-cols-4 md:grid-cols-6">
                    {originalImages.map((src, i) => (
                        <div key={i} className="group relative">
                            <img
                                src={src}
                                alt={`Image existante ${i + 1}`}
                                className="h-28 w-full rounded-md border object-cover"
                            />
                        </div>
                    ))}
                </div>
            ) : (
                <p className="mb-6 text-gray-500">Aucune image supplémentaire enregistrée.</p>
            )}
            <Dropzone
                multiple
                onDrop={onDrop}
                accept={{
                    "image/png": [".png"],
                    "image/jpeg": [".jpg", ".jpeg"],
                    "image/webp": [".webp"],
                }}
                maxFiles={10}>
                {({ getRootProps, getInputProps }) => (
                    <section>
                        <div
                            {...getRootProps()}
                            className="cursor-pointer rounded-md border-2 border-dashed border-gray-300 p-4 text-center hover:bg-gray-100">
                            <input {...getInputProps()} />
                            <p>Glissez-déposez vos photos, ou cliquez pour sélectionner. (max 10)</p>
                            <p className="text-xs text-gray-500">PNG, JPG, JPEG, WEBP (max 5 Mo)</p>
                        </div>
                    </section>
                )}
            </Dropzone>

            {pendingFiles.length > 0 && (
                <div>
                    <div className="mt-4 grid grid-cols-2 gap-4 sm:grid-cols-4 md:grid-cols-6">
                        {pendingFiles.map((file, i) => {
                            const preview = URL.createObjectURL(file);
                            return (
                                <div key={i} className="group relative">
                                    <img
                                        src={preview}
                                        alt={`Preview ${i}`}
                                        className="h-28 w-full rounded-md border object-cover"
                                    />
                                </div>
                            );
                        })}
                    </div>

                    <div className="mt-4 flex items-center gap-3">
                        <button
                            type="button"
                            onClick={handleUploadImages}
                            disabled={uploading || pendingFiles.length === 0}
                            className="rounded-md bg-gray-900 px-4 py-2 text-sm font-medium text-white hover:bg-black disabled:opacity-50">
                            {uploading ? "Synchronisation…" : "Synchroniser les images"}
                        </button>

                        <button
                            type="button"
                            onClick={() => {
                                pendingFiles.forEach((f) => URL.revokeObjectURL(f as any));
                                setPendingFiles([]);
                            }}
                            className="rounded-md border px-4 py-2 text-sm">
                            Annuler
                        </button>
                    </div>
                </div>
            )}
            {error && <p className="mt-2 text-sm text-red-600">{error}</p>}
        </div>
    );
}
